#!/usr/bin/env python3
# let's say you have a bucket on b2 containing videos which you've converted,
# using https://github.com/vincentbernat/video2hls, to a HLS file structure
# in subdirectories named for the sha1 sum of the original file.
# This script would produce an index of files available via HLS by parsing
# the output of `b2 ls BUCKET DIRECTORY --json --recursive` (available as
# data.json), returning json like
# [
#   ["d5db29cd03a2ed055086cef9c31c252b4587d6d0",
#    "filename.mp4"],
#   ...
# ]
# which could then be used to link to an HLS player like
# https://cdn.alyssadev.xyz/hls/#d5db29cd03a2ed055086cef9c31c252b4587d6d0
#
# data.json is loaded separately on each request because it's a fairly costly
# operation to generate a file listing of a large directory, so generation of
# that file might be best left to a separate process running infrequently.

from flask import *
# pip3 install b2
from b2sdk.v1 import SqliteAccountInfo, AuthInfoCache, B2RawApi, B2Http, B2Api
from json import load
import os

BUCKET = "ads-share"
HLS_PATH = "hls/"

app = Flask(__name__)

def get_api():
    info = SqliteAccountInfo()
    cache = AuthInfoCache(info)
    raw_api = B2RawApi(B2Http(
        user_agent_append=os.environ.get("B2_USER_AGENT_APPEND")
        ))
    return B2Api(info, cache=cache, raw_api=raw_api)

def get_hashes():
    api = get_api()
    bucket = api.get_bucket_by_name(BUCKET)
    generator = bucket.ls(HLS_PATH, recursive=True)
    for file_version_info, folder_name in generator:
        info = file_version_info.as_dict()
        h = info["fileName"].split("/")[1]
        if len(h) == 40:
            yield(h)

@app.route("/")
def index():
    with open("data.json") as f:
        data = {i["contentSha1"]: i for i in load(f) if "contentSha1" in i}
    hashes = set(get_hashes())
    out = [
            (data[h]["contentSha1"], data[h]["fileName"].split("/")[-1])
            for h in hashes]
    return jsonify(out)

if __name__ == "__main__":
    app.run(debug=True)