#!/usr/bin/env python3
from mutagen.easyid3 import EasyID3 # pypi mutagen
from parse import parse # pypi parse
from argparse import ArgumentParser
from glob import glob
from sys import exit

def write_tags(filename, newinfo):
    audio = EasyID3(filename)
    audio.update(newinfo)
    audio.save()
    return "{}: {}".format(filename, str(audio))

def main():
    parser = ArgumentParser()
    parser.add_argument("format", help="String format for input files e.g '{tracknumber} {title}.mp3'. Use '\"\" \"\" --print-keys' for valid keys")
    parser.add_argument("files", nargs="+", help="Filenames to process. Accepts shell wildcard expansion")
    parser.add_argument("--artist", help="Artist name to add to track tags, overriding filename")
    parser.add_argument("--album", help="Album name to add to track tags, overriding filename")
    parser.add_argument("--print-keys", help="Prints valid keys for string format via mutagen. You'll need to add \"\" \"\" to satisfy the missing format/files parameters.", action="store_true")
    argv = parser.parse_args()
    if argv.print_keys:
        print(EasyID3.valid_keys.keys())
        return 0
    if len(argv.files) == 1:
        argv.files = glob(argv.files[0])
    for f in argv.files:
        newinfo = parse(argv.format, f)
        if not newinfo:
            print("Error parsing {}: Format does not match filename".format(f))
            continue
        newinfo = newinfo.named
        if argv.artist: newinfo["artist"] = argv.artist
        if argv.album:  newinfo["album"]  = argv.album
        print(write_tags(f, newinfo))
    return 0

if __name__ == "__main__":
    try:
        exit(main())
    except KeyboardInterrupt:
        exit(130)