#!/bin/bash
# Provisions a droplet on DigitalOcean to download a given torrent file, then reupload to a given rclone remote
# Prints estimated USD cost of transaction when complete
# uses your user's rclone.conf, so set up the remote locally and make sure it works
# torrentdl.sh https://releases.ubuntu.com/20.04/ubuntu-20.04.1-live-server-amd64.iso.torrent b2:my-linux-isos/ubuntu/
export IMAGE="ubuntu-20-04-x64"
export REGION="sgp1"
export SIZE="s-4vcpu-8gb"
export HOSTNAME="torrent-dl"
export TORRENT="$1"
export DESTREMOTE="$2"
export URLREGEX='^(https?|ftp|file)://[-A-Za-z0-9\+&@#/%?=~_|!:,.;]*[-A-Za-z0-9\+&@#/%=~_|]$'

if [[ $TORRENT =~ $URLREGEX ]]; then
        echo Downloading torrent from url...
        wget "$TORRENT"
        export TORRENT=`basename "$TORRENT"`
fi

echo "Downloading '$TORRENT' on $REGION:$IMAGE:$SIZE with hostname $HOSTNAME, then uploading to '$DESTREMOTE'"
read -p "Press enter to continue, ctrl-c to abort..."

echo "Started at `date`"
export STARTTIME=`date +%s`

echo "Provisioning droplet..."
export SSHKEYS=`doctl compute ssh-key list --no-header --format ID | tr '\n' ',' | sed 's/,$/\n/'`
export IP=$( set -x; doctl compute droplet create $HOSTNAME --ssh-keys $SSHKEYS --region $REGION --size $SIZE --image $IMAGE --wait --no-header --format PublicIPv4 )

echo -n "Droplet at $IP, waiting for connection"
until (ssh -o StrictHostKeyChecking=no root@$IP mkdir -p /root/.config/rclone /root/t 2>/dev/null >/dev/null); do
        echo -n "."
        sleep 1
done
ssh-keyscan -H $IP 2>/dev/null >> ~/.ssh/known_hosts
echo

echo Copying rclone conf and torrent to droplet...
scp ~/.config/rclone/rclone.conf root@$IP:/root/.config/rclone/rclone.conf
scp "$TORRENT" root@$IP:/root/t/

echo Installing dependencies...
ssh -t root@$IP "apt-get update; apt-get install -y aria2; curl https://rclone.org/install.sh | sudo bash"
echo Downloading torrent...
ssh -t root@$IP aria2c -d "/root/t/" --enable-dht=false --enable-peer-exchange=false --seed-time=0 "/root/t/*.torrent"
echo Uploading torrent to rclone remote...
ssh -t root@$IP rclone copy /root/t/ "$DESTREMOTE" -P --fast-list --transfers=$(($LINES-5))
echo Job complete, deleting droplet
$(set -x; doctl compute droplet delete -f $HOSTNAME)

echo "Finished at `date`"
export ENDTIME=`date +%s`
echo "Total run time: $(($ENDTIME-$STARTTIME))"
echo -n "Total (estimated) cost: USD $"
doctl compute size list | grep $SIZE | awk '{print $5}' | while read p; do python3 -c "print(($p/2592000)*($ENDTIME-$STARTTIME))"; done