#!/usr/bin/env python
from __future__ import print_function
import sys

try:
    import requests
    from bs4 import BeautifulSoup as Soup
    from Crypto.PublicKey import RSA
    from Crypto.Cipher import PKCS1_v1_5
    import yaml
except ImportError:
    print("Please run \"pip install requests beautifulsoup4 pycrypto pyyaml\"", file=sys.stderr)
    sys.exit(1)

import time
import json
import base64
import os

H = {"User-Agent": "Mozilla/4.0 (compatible; MSIE 5.5; Windows NT)"}
session = requests.Session()
session.headers.update(H)

def get_key(username):
    return session.post("https://steamcommunity.com/login/getrsakey/",
        data=dict(username=username, donotcache=time.time()*1000) ).json()

def encode_passwd(key_data, passwd):
    mod = int(key_data["publickey_mod"], 16)
    exp = int(key_data["publickey_exp"], 16)
    rsa = RSA.construct((mod, exp))
    cipher = PKCS1_v1_5.new(rsa)
    if hasattr(passwd, "encode"):
        passwd = passwd.encode("utf-8")
    return base64.b64encode(cipher.encrypt(passwd))

def login(username, password):
    key = get_key(username)
    encrypted_password = encode_passwd(key, password)
    return session.post("https://steamcommunity.com/login/dologin/",
        data=dict(
            username=username,
            password=encrypted_password,
            emailauth="",
            loginfriendlyname="",
            captchagid="-1",
            captcha_text="",
            emailsteamid="",
            rsatimestamp=key["timestamp"],
            remember_login=True,
            donotcache=time.time()*1000
        ) ).json()

def transfer(url, login_data):
    session.post(url,
        data=login_data["transfer_parameters"]
    )

def get_gameservers():
    """Yields lists containing four items: steam game id, token, date last used and memo"""
    page = Soup(session.get("https://steamcommunity.com/dev/managegameservers").text, "html.parser")
    response = []
    for row in page.findAll("tr")[1:]:
        response.append([ i.text.replace("\\", "\\\\").replace('"', '\\"') for i in row.findAll("td")[:4] ])
    return response

if __name__ == "__main__":
    username = os.environ.get("STEAM_USERNAME")
    password = os.environ.get("STEAM_PASSWORD")
    if username and password:
        if os.path.isfile("/tmp/steam-{}.json".format(username)):
            with open("/tmp/steam-{}.json".format(username)) as f:
                session.cookies = requests.utils.cookiejar_from_dict(json.loads(f.read()))
        else:
            login_data = login(username, password)
            for url in login_data["transfer_urls"]:
                transfer(url, login_data)
        for gs in get_gameservers():
            if "(expired)" in gs[1]:
                gs[1] = gs[1][:32]
                gs.append(True)
            else:
                gs.append(False)
            print(yaml.safe_dump( {gs[3]: dict(id=gs[0], token=gs[1], date=gs[2], expired=gs[4])}, default_flow_style=False ).strip())
        with open("/tmp/steam-{}.json".format(username), "w") as f:
            f.write(json.dumps(requests.utils.dict_from_cookiejar(session.cookies)))
    else:
        print("Please specify environment variables STEAM_USERNAME and STEAM_PASSWORD", file=sys.stderr)