#!/usr/bin/env python3
__usage__ = """
usage: torrenttags [-h] [--html] hash

positional arguments:
  hash        Torrent hash to look up

optional arguments:
  -h, --help  show this help message and exit
  --html      Prints html response to stdout, includes Chilling Effects
              reports if available
"""

from requests import get
from random import choice
from bs4 import BeautifulSoup as Soup
from argparse import ArgumentParser
from sys import exit, stderr
from base64 import b16encode, b32decode

urls = ["http://api1.torrenttags.com/v1/", "http://api2.torrenttags.com/v1/", "http://api3.torrenttags.com/v1/"]

def get_api_url(method=None):
    return choice(urls) + method

def get_ticket():
    data = get(get_api_url("get-ticket")).json()
    return data["status"], data["ticket"]

def get_result(hash, ticket=None, rethtml=False):
    if not ticket:
        ticket = get_ticket()
    if len(hash) != 40:
        hash = convert_hash(hash)
    data = Soup(get(get_api_url("get-tags-html"), params={'ticket': ticket, 'torrent': hash}).json()["html"], "html.parser")
    if "no_claims" in data.find('img')["src"]:
        return True, data
    return False, data

def convert_hash(torrent_hash):
    """ Turns out hashes in magnet links can be base32 encoded, which shortens them to 32 characters """
    b16 = b16encode(b32decode(torrent_hash))
    return b16.decode('utf-8').lower()

def main():
    parser = ArgumentParser()
    parser.add_argument("hash", help="Torrent hash to look up")
    parser.add_argument("--html", help="Prints html response to stdout, includes Chilling Effects reports if available", action="store_true")
    args = parser.parse_args()
    result, data = get_result(args.hash, rethtml=args.html)
    if args.html:
        print(data)
    if result:
        return "{}: No claims (yet!)".format(args.hash), 0
    else:
        return "{}: Claims found".format(args.hash), 2

if __name__ == "__main__":
    outp, ret = main()
    print(outp, file=stderr)
    exit(ret)
