#!/usr/bin/env python3
from gmusicapi import Mobileclient # pip3 install gmusicapi
import os
from json import load
import sys

def mkdir_p(path):
    import errno
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise

mkdir_p(os.path.expanduser("~/.config"))
with open(os.path.expanduser("~/.config/gmusic.json")) as f:
    CONFIG = load(f)

client = Mobileclient()
client.login(*CONFIG)

def search(term, results=10):
    results = client.search(term, max_results=results)
    return {"album": [client.get_album_info(x["album"]["albumId"]) for x in results["album_hits"]],
            "artist": [client.get_artist_info(x["artist"]["artistId"], max_rel_artist=0) for x in results["artist_hits"]],
            "playlist": [client.get_shared_playlist_contents(x["playlist"]["shareToken"]) for x in results["playlist_hits"]],
            "podcast": [client.get_podcast_series_info(x["series"]["seriesId"]) for x in results["podcast_hits"]],
            "song": [x["track"] for x in results["song_hits"]],
            "video": [x.get("youtube_video", None) for x in results["video_hits"]],
           }

def get_stream(results, selection):
    section, _, n = selection.partition("#")
    if not section in results:
        raise ValueError("Invalid section name")
    n = int(n)
    if len(results[section]) > n:
        raise ValueError("Invalid number")
    result = results[section][n-1]
    if section == "song":
        return client.get_stream_url(result.get("storeId") or result.get("id"))
    else:
        raise NotImplemented

def main():
    from argparse import ArgumentParser
    parser = ArgumentParser()
    parser.add_argument("term", nargs="+")
    parser.add_argument("-n", help="number of results returned", type=int, default=10)
    args = parser.parse_args()
    results = search(args.term, results=args.n)
    for section in results:
        print("\n".join("{}#{}: {}".format(section, n+1, i["title"]) for n,i in enumerate(results[section]) if type(i) is dict and i.get("title")), file=sys.stderr)
    print("Selection (section#number): ", end="", file=sys.stderr)
    print(get_stream(results, input()))
    return 0

if __name__ == "__main__":
    sys.exit(main())
    
